library(readxl)
library(dplyr)
library(tidyr)
library(ggplot2)
library(scales)

# ---- Read Fisher.xlsx ----
df <- read_excel("Fisher.xlsx") |>
  transmute(
    Test_groups = Group,
    `good flight` = `Good flight`,
    `bad flight`  = `Bad flight`
  )

# ---- Long format + percentages ----
dlong <- df |>
  pivot_longer(cols = c(`good flight`, `bad flight`),
               names_to = "Experimental_condition",
               values_to = "Count") |>
  group_by(Test_groups) |>
  mutate(
    Total = sum(Count),
    Prop  = Count / Total,
    Pct   = round(100 * Prop),
    ypos  = cumsum(Prop) - 0.5 * Prop,
    label = paste0(Experimental_condition, "\n", Pct, "%")
  ) |>
  ungroup()

# keep good flight on top
dlong$Experimental_condition <- factor(
  dlong$Experimental_condition,
  levels = c("bad flight", "good flight")
)

# ---- Overall test (3+ groups) ----
mat <- as.matrix(df[, c("bad flight", "good flight")])
rownames(mat) <- df$Test_groups

chisq_res <- chisq.test(mat, correct = FALSE)

# ---- Plot ----
p <- ggplot(dlong, aes(x = Test_groups, y = Prop, fill = Experimental_condition)) +
  geom_col(width = 0.85) +
  geom_label(
    aes(y = ypos, label = label),
    size = 5.0,
    fontface = "bold",
    fill = "white",
    label.size = 0.3,
    colour = "black",
    lineheight = 0.95
  ) +
  scale_y_continuous(labels = percent_format(accuracy = 1), limits = c(0, 1)) +
  labs(
    title = paste0(
      "Chi-square test (overall), p-value = ",
      ifelse(chisq_res$p.value < 0.001, "< 0.001", round(chisq_res$p.value, 3))
    ),
    x = "Test_groups",
    y = NULL,
    fill = "Experimental_condition"
  ) +
  theme_classic(base_size = 14) +
  theme(
    plot.title = element_text(face = "bold"),
    legend.title = element_text(face = "bold"),
    axis.text.x = element_text(size = 13),
    plot.margin = margin(12, 20, 12, 12)
  )

p

########
library(combinat)

groups <- df$Test_groups
pairs <- combn(groups, 2, simplify = FALSE)

pairwise_res <- lapply(pairs, function(g) {
  sub <- df |> filter(Test_groups %in% g)
  
  mat <- matrix(
    c(sub$`bad flight`, sub$`good flight`),
    nrow = 2, byrow = FALSE,
    dimnames = list(g, c("bad flight", "good flight"))
  )
  
  ft <- fisher.test(mat)
  
  data.frame(
    group1 = g[1],
    group2 = g[2],
    p_value = ft$p.value
  )
})

pairwise_df <- bind_rows(pairwise_res) |>
  mutate(p_adj_BH = p.adjust(p_value, method = "BH"))

pairwise_df

